/* PUE · Agendamento do Cidadão — Demo (React + Tailwind via CDN)
   v2: Login OIDC simulado + alternância Demo/Real + Base URL persistente
*/

// ---------------- Mock Data ----------------
const services = [
  { id: "svc-registos", nome: "Registos e Notariado", requisitos: "Documento de identificação válido" },
  { id: "svc-financas", nome: "Finanças", requisitos: "NIF e situação fiscal regularizada" },
  { id: "svc-ss", nome: "Segurança Social", requisitos: "NISS e comprovativo de situação" },
  { id: "svc-imt", nome: "IMT / Carta de Condução", requisitos: "Exames e documentos originais" }
];
const sites = [
  { id: "lx-sede", nome: "Lisboa — Sede Central", morada: "Av. República, 100", serviceId: "svc-registos" },
  { id: "lx-norte", nome: "Lisboa — Polo Norte", morada: "R. do Norte, 45", serviceId: "svc-financas" },
  { id: "pt-aliados", nome: "Porto — Aliados", morada: "Av. dos Aliados, 10", serviceId: "svc-registos" },
  { id: "pt-campanha", nome: "Porto — Campanhã", morada: "Est. de Campanhã, 5", serviceId: "svc-ss" },
  { id: "coimbra-centro", nome: "Coimbra — Centro", morada: "Largo da Portagem", serviceId: "svc-imt" }
];

// ---------------- Utilitários ----------------
function pad(n){ return n<10?`0${n}`:`${n}`; }
function fmtDate(d){ return `${d.getFullYear()}-${pad(d.getMonth()+1)}-${pad(d.getDate())}`; }
function fmtHuman(dt){ return `${pad(dt.getDate())}/${pad(dt.getMonth()+1)}/${dt.getFullYear()} ${pad(dt.getHours())}:${pad(dt.getMinutes())}`; }
function uid(prefix="id"){ return `${prefix}_${Math.random().toString(36).slice(2,8)}${Date.now().toString(36).slice(-4)}`; }

function genSlots(siteId, serviceId, dateISO){
  const base = new Date(dateISO + "T09:00:00");
  const out = [];
  for (let i=0; i<16; i++){
    const start = new Date(base.getTime() + i * 30 * 60000);
    const end = new Date(start.getTime() + 25 * 60000);
    const id = `${siteId}-${serviceId}-${i}`;
    const taken = Math.floor((Math.sin(i * 1.7) + 1) * 1.2);
    out.push({ id, siteId, serviceId, start: start.toISOString(), end: end.toISOString(), capacity: 3, taken });
  }
  return out;
}

// ---------------- Storage ----------------
function saveBooking(b){
  const key="pue_bookings";
  const arr = JSON.parse(localStorage.getItem(key) || "[]");
  arr.push(b); localStorage.setItem(key, JSON.stringify(arr));
}
function listBookings(){ return JSON.parse(localStorage.getItem("pue_bookings") || "[]"); }

function saveSession(sess){
  if(sess) localStorage.setItem("pue_session", JSON.stringify(sess));
  else localStorage.removeItem("pue_session");
}
function loadSession(){
  try { return JSON.parse(localStorage.getItem("pue_session")||"null"); } catch(e){ return null; }
}
function saveMode(mode){ localStorage.setItem("pue_mode", mode); }
function loadMode(){ return localStorage.getItem("pue_mode") || "demo"; }
function saveBaseURL(u){ localStorage.setItem("pue_baseurl", u); }
function loadBaseURL(){ return localStorage.getItem("pue_baseurl") || "http://localhost:8080"; }

// ---------------- API Layer ----------------
const apiDemo = {
  async listServices(){ return services; },
  async listSitesByService(serviceId){ return sites.filter(s=>s.serviceId===serviceId); },
  async listSlots(serviceId, siteId, dateISO){ return genSlots(siteId, serviceId, dateISO); },
  async createAppointment(payload){
    // simulate server create by echoing id
    return { id: uid("appt"), ...payload };
  }
};

function makeApiReal(baseURL){
  async function safeFetch(path, opts){
    try{
      const r = await fetch(baseURL + path, { ...opts, headers: { "Content-Type":"application/json", ...(opts&&opts.headers||{}) } });
      if(!r.ok) throw new Error("HTTP "+r.status);
      return await r.json();
    }catch(e){
      console.warn("[API Real] falhou", path, e);
      return null;
    }
  }
  return {
    async listServices(){
      const data = await safeFetch("/v1/services");
      return data || apiDemo.listServices();
    },
    async listSitesByService(serviceId){
      const data = await safeFetch(`/v1/sites?service_id=${encodeURIComponent(serviceId)}`);
      if(data && Array.isArray(data) && data.length) return data;
      // fallback: filtrar mocks
      return apiDemo.listSitesByService(serviceId);
    },
    async listSlots(serviceId, siteId, dateISO){
      const data = await safeFetch(`/v1/slots?service_id=${encodeURIComponent(serviceId)}&site_id=${encodeURIComponent(siteId)}&date=${encodeURIComponent(dateISO)}`);
      return (data && Array.isArray(data) && data.length) ? data : apiDemo.listSlots(serviceId, siteId, dateISO);
    },
    async createAppointment(payload){
      const data = await safeFetch(`/v1/appointments`, { method:"POST", body: JSON.stringify(payload) });
      return data || apiDemo.createAppointment(payload);
    }
  };
}

function mkICS(b, s, site, slot){
  const dtStart = new Date(slot.start);
  const dtEnd = new Date(slot.end);
  const dtStamp = new Date();
  const toICS = d => d.toISOString().replace(/[-:]/g,'').split('.')[0]+'Z';
  const ics = [
    "BEGIN:VCALENDAR",
    "VERSION:2.0",
    "PRODID:-//PUE//Agendamento do Cidadao//PT",
    "CALSCALE:GREGORIAN",
    "BEGIN:VEVENT",
    `UID:${b.id}@pue.gov.pt`,
    `DTSTAMP:${toICS(dtStamp)}`,
    `DTSTART:${toICS(dtStart)}`,
    `DTEND:${toICS(dtEnd)}`,
    `SUMMARY:Agendamento — ${s.nome}`,
    `LOCATION:${site.nome}, ${site.morada}`,
    `DESCRIPTION:Agendamento ${b.id} — Cidadão: ${b.citizen}`,
    "END:VEVENT",
    "END:VCALENDAR"
  ].join("\\r\\n");
  const blob = new Blob([ics], { type: "text/calendar;charset=utf-8" });
  return URL.createObjectURL(blob);
}

// ---------------- UI Primitives ----------------
function Pill({children, active, onClick}){
  return <button onClick={onClick} className={`px-3 py-1 rounded-full border text-sm mr-2 mb-2 ${active?'bg-[#ffcc66] text-black border-[#ffcc66]':'border-zinc-600 text-zinc-200 hover:border-zinc-400'}`}>{children}</button>;
}
function Card({title, children, right}){
  return (
    <div className="rounded-2xl bg-[#0f111a] border border-zinc-800 shadow-md p-4">
      {title && (
        <div className="flex items-center justify-between mb-3">
          <h3 className="text-[#ffcc66] text-lg font-semibold">{title}</h3>
          {right}
        </div>
      )}
      {children}
    </div>
  );
}
function Step({n, label, active, done}){
  return (
    <div className="flex items-center">
      <div className={`w-8 h-8 rounded-full flex items-center justify-center mr-2 ${done?'bg-emerald-500 text-black':active?'bg-[#ffcc66] text-black':'bg-zinc-700 text-zinc-200'}`}>{n}</div>
      <span className={`mr-4 text-sm ${active?'text-white':'text-zinc-400'}`}>{label}</span>
    </div>
  );
}
function Modal({open, title, children, onClose}){
  if(!open) return null;
  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center">
      <div className="absolute inset-0 bg-black/60" onClick={onClose}/>
      <div className="relative z-10 w-[95%] max-w-md rounded-2xl bg-[#0f111a] border border-zinc-700 p-4 shadow-xl">
        <div className="flex items-center justify-between mb-3">
          <h3 className="text-[#ffcc66] text-lg font-semibold">{title}</h3>
          <button onClick={onClose} className="text-zinc-400 hover:text-zinc-200">✕</button>
        </div>
        {children}
      </div>
    </div>
  );
}

// ---------------- App ----------------
function App(){
  const [tab, setTab] = React.useState('marcar');

  // Sessão simulada OIDC
  const [session, setSession] = React.useState(loadSession());
  const logged = !!session;
  const citizenDefault = session?.name || "Cidadão Demo";
  const emailDefault = session?.email || "cidadao@exemplo.pt";

  // Modo Demo/Real + Base URL
  const [mode, setMode] = React.useState(loadMode()); // "demo" | "real"
  const [baseURL, setBaseURL] = React.useState(loadBaseURL());
  const api = React.useMemo(()=> mode === "real" ? makeApiReal(baseURL) : apiDemo, [mode, baseURL]);

  // Login modal
  const [loginOpen, setLoginOpen] = React.useState(false);
  const [loginName, setLoginName] = React.useState(citizenDefault);
  const [loginEmail, setLoginEmail] = React.useState(emailDefault);
  const [loginMethod, setLoginMethod] = React.useState("CC/Chave Móvel (simulado)");

  function doLogin(){
    const sess = {
      sub: uid("sub"),
      name: loginName.trim() || "Cidadão Demo",
      email: loginEmail.trim() || "cidadao@exemplo.pt",
      method: loginMethod,
      id_token: uid("idtok"), // simulado
      iat: Date.now()
    };
    saveSession(sess); setSession(sess); setLoginOpen(false);
  }
  function doLogout(){
    saveSession(null); setSession(null);
  }

  React.useEffect(()=> saveMode(mode), [mode]);
  React.useEffect(()=> saveBaseURL(baseURL), [baseURL]);

  // Stepper e estado da marcação
  const [step, setStep] = React.useState(1);
  const [serviceId, setServiceId] = React.useState(services[0].id);
  const [siteId, setSiteId] = React.useState("");
  const [dateISO, setDateISO] = React.useState(fmtDate(new Date(Date.now() + 86400000)));
  const [slots, setSlots] = React.useState([]);
  const [slotId, setSlotId] = React.useState("");
  const [notes, setNotes] = React.useState("");
  const [confirmed, setConfirmed] = React.useState(null);

  const svc = React.useMemo(()=> services.find(s=>s.id===serviceId), [serviceId]);
  const siteOpts = React.useMemo(()=> sites.filter(s=>s.serviceId===serviceId), [serviceId]);
  const site = React.useMemo(()=> siteOpts.find(s=>s.id===siteId) || null, [siteId, siteOpts]);
  const slot = React.useMemo(()=> slots.find(s=>s.id===slotId) || null, [slotId, slots]);

  React.useEffect(()=>{ if(siteOpts.length && !siteId) setSiteId(siteOpts[0].id); }, [siteOpts, siteId]);
  React.useEffect(()=>{ (async()=>{ if(serviceId && siteId && dateISO){ const data = await api.listSlots(serviceId, siteId, dateISO); setSlots(data); } })(); }, [serviceId, siteId, dateISO, api]);

  const canNext1 = !!serviceId;
  const canNext2 = !!siteId;
  const canNext3 = !!dateISO;
  const canNext4 = !!slotId;

  async function onConfirm(){
    if(!svc || !site || !slot) return;
    const payload = {
      serviceId: svc.id, siteId: site.id, date: dateISO, slotId: slot.id,
      citizen: session?.name || "Cidadão Demo", email: session?.email || "cidadao@exemplo.pt", notes
    };
    const created = await api.createAppointment(payload);
    const b = { id: created.id || uid("appt"), ...payload };
    saveBooking(b); setConfirmed(b); setTab('meus');
  }

  const myBookings = React.useMemo(()=> listBookings().sort((a,b)=> a.date.localeCompare(b.date)), [confirmed, tab]);
  const ocupacao = React.useMemo(()=>{
    if(!slots.length) return 0;
    const tot = slots.reduce((acc,s)=> acc+s.capacity, 0);
    const used = slots.reduce((acc,s)=> acc+Math.min(s.taken, s.capacity), 0);
    return Math.round((used/tot)*100);
  }, [slots]);

  return (
    <div className="min-h-screen bg-[#0b0b1a] text-[#e8eef4]">
      {/* Header */}
      <header className="border-b border-zinc-900 bg-[#0d0f17]/80 backdrop-blur sticky top-0 z-40">
        <div className="max-w-6xl mx-auto px-4 py-4 flex items-center justify-between">
          <div className="flex items-center gap-3">
            <div className="w-9 h-9 rounded-xl bg-[#ffcc66] text-black font-bold flex items-center justify-center">P</div>
            <div>
              <div className="text-[#ffcc66] font-semibold leading-none">Plataforma Unificada do Estado</div>
              <div className="text-xs text-zinc-400">Agendamento do Cidadão — Demo</div>
            </div>
          </div>
          <div className="flex items-center gap-2">
            <div className="hidden md:flex items-center gap-2 mr-2">
              <span className="text-xs text-zinc-400">Modo:</span>
              <button onClick={()=> setMode(mode==="demo"?"real":"demo")} className={`px-2 py-1 rounded-full border text-xs ${mode==="demo"?'border-zinc-600 text-zinc-200':'bg-[#ffcc66] text-black border-[#ffcc66]'}`}>
                {mode==="demo" ? "Dados Demo" : "Dados Reais"}
              </button>
            </div>
            <div className="px-3 py-1 rounded-full bg-zinc-800 text-sm">{logged? session.name : "Anónimo"}</div>
            {!logged ? (
              <button onClick={()=> setLoginOpen(true)} className="px-3 py-1 rounded-xl bg-[#ffcc66] text-black">Entrar</button>
            ) : (
              <button onClick={doLogout} className="px-3 py-1 rounded-xl bg-zinc-800 text-zinc-200">Sair</button>
            )}
          </div>
        </div>
        <div className="max-w-6xl mx-auto px-4 pb-3 flex items-center gap-2">
          <Pill active={tab==='marcar'} onClick={()=>setTab('marcar')}>Marcar</Pill>
          <Pill active={tab==='meus'} onClick={()=>setTab('meus')}>Os meus agendamentos</Pill>
          <Pill active={tab==='admin'} onClick={()=>setTab('admin')}>Admin (demo)</Pill>
        </div>
      </header>

      {/* Banner modo real */}
      {mode==="real" && (
        <div className="bg-amber-900/40 border-b border-amber-700 text-amber-200">
          <div className="max-w-6xl mx-auto px-4 py-2 text-sm">
            Modo <b>Dados Reais</b> activo — a usar <span className="font-mono">{baseURL}</span>. Em caso de falha, recorre a dados Demo.
          </div>
        </div>
      )}

      <main className="max-w-6xl mx-auto px-4 py-6 grid gap-6 lg:grid-cols-3">
        {/* Coluna principal */}
        <div className="lg:col-span-2 grid gap-6">
          {tab==='marcar' && (
            <Card>
              <div className="flex flex-wrap items-center gap-3 mb-4">
                <Step n={1} label="Serviço" active={step===1} done={step>1} />
                <Step n={2} label="Local" active={step===2} done={step>2} />
                <Step n={3} label="Data" active={step===3} done={step>3} />
                <Step n={4} label="Slot" active={step===4} done={step>4} />
                <Step n={5} label="Confirmação" active={step===5} done={false} />
              </div>

              {step===1 && (
                <div className="grid gap-3">
                  <label className="text-sm text-zinc-300">Escolha o serviço</label>
                  <select value={serviceId} onChange={e=>setServiceId(e.target.value)} className="bg-zinc-900 border border-zinc-700 rounded-xl px-3 py-2">
                    {services.map(s => <option key={s.id} value={s.id}>{s.nome}</option>)}
                  </select>
                  {svc?.requisitos && <div className="text-xs text-zinc-400">Requisitos: {svc.requisitos}</div>}
                  <div className="flex justify-end mt-2">
                    <button disabled={!serviceId} onClick={()=>setStep(2)} className={`px-4 py-2 rounded-xl ${serviceId?'bg-[#ffcc66] text-black':'bg-zinc-800 text-zinc-500 cursor-not-allowed'}`}>Seguinte</button>
                  </div>
                </div>
              )}

              {step===2 && (
                <div className="grid gap-3">
                  <label className="text-sm text-zinc-300">Escolha o local/posto de atendimento</label>
                  <select value={siteId} onChange={e=>setSiteId(e.target.value)} className="bg-zinc-900 border border-zinc-700 rounded-xl px-3 py-2">
                    {siteOpts.map(s => <option key={s.id} value={s.id}>{s.nome}</option>)}
                  </select>
                  {site && <div className="text-xs text-zinc-400">Morada: {site.morada}</div>}
                  <div className="flex justify-between mt-2">
                    <button onClick={()=>setStep(1)} className="px-4 py-2 rounded-xl bg-zinc-800 text-zinc-200">Anterior</button>
                    <button disabled={!siteId} onClick={()=>setStep(3)} className={`px-4 py-2 rounded-xl ${siteId?'bg-[#ffcc66] text-black':'bg-zinc-800 text-zinc-500 cursor-not-allowed'}`}>Seguinte</button>
                  </div>
                </div>
              )}

              {step===3 && (
                <div className="grid gap-3">
                  <label className="text-sm text-zinc-300">Escolha a data</label>
                  <input type="date" value={dateISO} onChange={e=>setDateISO(e.target.value)} className="bg-zinc-900 border border-zinc-700 rounded-xl px-3 py-2 w-56" />
                  <div className="flex justify-between mt-2">
                    <button onClick={()=>setStep(2)} className="px-4 py-2 rounded-xl bg-zinc-800 text-zinc-200">Anterior</button>
                    <button disabled={!dateISO} onClick={()=>setStep(4)} className={`px-4 py-2 rounded-xl ${dateISO?'bg-[#ffcc66] text-black':'bg-zinc-800 text-zinc-500 cursor-not-allowed'}`}>Ver slots</button>
                  </div>
                </div>
              )}

              {step===4 && (
                <div className="grid gap-4">
                  <div className="flex items-center justify-between">
                    <div className="text-sm text-zinc-300">Slots disponíveis em <span className="text-white font-medium">{site?.nome}</span> — {dateISO}</div>
                    <button onClick={()=>setStep(3)} className="px-3 py-1 rounded-xl bg-zinc-800 text-zinc-200">Alterar data</button>
                  </div>
                  <div className="grid grid-cols-2 md:grid-cols-3 lg:grid-cols-4 gap-3">
                    {slots.map(s=>{
                      const free = Math.max(s.capacity - s.taken, 0);
                      const selected = s.id === slotId;
                      return (
                        <button key={s.id} onClick={()=>setSlotId(s.id)} className={`rounded-xl border p-3 text-left ${selected?'border-[#ffcc66] bg-zinc-900':'border-zinc-700 bg-zinc-950 hover:border-zinc-500'}`}>
                          <div className="text-sm">{fmtHuman(new Date(s.start)).slice(11,16)} — {fmtHuman(new Date(s.end)).slice(11,16)}</div>
                          <div className={`text-xs mt-1 ${free>0?'text-emerald-400':'text-rose-400'}`}>{free>0? `${free} vagas` : 'Sem vagas'}</div>
                        </button>
                      );
                    })}
                  </div>
                  <div className="grid gap-2">
                    <label className="text-sm text-zinc-300">Notas (opcional)</label>
                    <textarea value={notes} onChange={e=>setNotes(e.target.value)} placeholder="Ex.: Levar documentos originais" className="bg-zinc-900 border border-zinc-700 rounded-xl px-3 py-2 min-h-[80px]" />
                  </div>
                  <div className="flex justify-between mt-2">
                    <button onClick={()=>setStep(3)} className="px-4 py-2 rounded-xl bg-zinc-800 text-zinc-200">Anterior</button>
                    <button disabled={!slotId} onClick={()=>setStep(5)} className={`px-4 py-2 rounded-xl ${slotId?'bg-[#ffcc66] text-black':'bg-zinc-800 text-zinc-500 cursor-not-allowed'}`}>Prosseguir</button>
                  </div>
                </div>
              )}

              {step===5 && (
                <div className="grid gap-3">
                  <div className="text-sm text-zinc-300">Confirmar dados do agendamento</div>
                  <div className="grid grid-cols-1 md:grid-cols-2 gap-3">
                    <div className="rounded-xl border border-zinc-700 p-3 bg-zinc-950">
                      <div className="text-xs text-zinc-400">Serviço</div>
                      <div className="font-medium">{svc?.nome}</div>
                    </div>
                    <div className="rounded-xl border border-zinc-700 p-3 bg-zinc-950">
                      <div className="text-xs text-zinc-400">Local</div>
                      <div className="font-medium">{site?.nome}</div>
                    </div>
                    <div className="rounded-xl border border-zinc-700 p-3 bg-zinc-950">
                      <div className="text-xs text-zinc-400">Data</div>
                      <div className="font-medium">{dateISO}</div>
                    </div>
                    <div className="rounded-xl border border-zinc-700 p-3 bg-zinc-950">
                      <div className="text-xs text-zinc-400">Hora</div>
                      <div className="font-medium">{slot? fmtHuman(new Date(slot.start)).slice(11,16) : '-'}</div>
                    </div>
                  </div>
                  <div className="flex justify-between mt-2">
                    <button onClick={()=>setStep(4)} className="px-4 py-2 rounded-xl bg-zinc-800 text-zinc-200">Anterior</button>
                    <button onClick={onConfirm} className="px-4 py-2 rounded-xl bg-[#ffcc66] text-black">Confirmar agendamento</button>
                  </div>
                </div>
              )}
            </Card>
          )}

          {tab==='meus' && (
            <Card title="Os meus agendamentos">
              {confirmed && (
                <div className="mb-4 p-3 rounded-xl bg-emerald-900/30 border border-emerald-700 text-emerald-200">
                  ✅ Agendamento confirmado: <span className="font-mono">{confirmed.id}</span>
                </div>
              )}
              <div className="grid gap-3">
                {listBookings().length===0 && <div className="text-zinc-400">Sem agendamentos. Faça a sua marcação na aba “Marcar”.</div>}
                {listBookings().sort((a,b)=> a.date.localeCompare(b.date)).map(b=>{
                  const s = services.find(x=>x.id===b.serviceId);
                  const st = sites.find(x=>x.id===b.siteId);
                  const sl = genSlots(b.siteId, b.serviceId, b.date).find(x=>x.id===b.slotId);
                  const url = mkICS(b, s, st, sl);
                  return (
                    <div key={b.id} className="rounded-xl border border-zinc-700 p-3 bg-zinc-950 flex items-center justify-between">
                      <div>
                        <div className="text-[#ffcc66] font-medium">{s.nome}</div>
                        <div className="text-sm text-zinc-300">{st.nome} — {fmtHuman(new Date(sl.start))}</div>
                        <div className="text-xs text-zinc-500">Código: <span className="font-mono">{b.id}</span></div>
                      </div>
                      <div className="flex items-center gap-2">
                        <a href={url} download={`agendamento_${b.id}.ics`} className="px-3 py-2 rounded-xl bg-zinc-800 hover:bg-zinc-700">Exportar ICS</a>
                        <button onClick={()=>navigator.clipboard.writeText(b.id)} className="px-3 py-2 rounded-xl bg-zinc-800 hover:bg-zinc-700">Copiar código</button>
                      </div>
                    </div>
                  );
                })}
              </div>
            </Card>
          )}

          {tab==='admin' && (
            <Card title="Admin (demo) — Operação do dia" right={<span className="text-xs text-zinc-400">{fmtDate(new Date())}</span>}>
              <div className="grid md:grid-cols-3 gap-3">
                <div className="rounded-xl border border-zinc-700 p-3 bg-zinc-950">
                  <div className="text-xs text-zinc-400">Ocupação média</div>
                  <div className="text-3xl font-semibold">{ocupacao}%</div>
                  <div className="h-2 mt-2 rounded bg-zinc-800">
                    <div className="h-2 rounded bg-[#ffcc66]" style={{ width: `${ocupacao}%` }} />
                  </div>
                </div>
                <div className="rounded-xl border border-zinc-700 p-3 bg-zinc-950">
                  <div className="text-xs text-zinc-400">Slots (dia)</div>
                  <div className="text-3xl font-semibold">{slots.length}</div>
                </div>
                <div className="rounded-xl border border-zinc-700 p-3 bg-zinc-950">
                  <div className="text-xs text-zinc-400">Serviço seleccionado</div>
                  <div className="text-lg">{svc?.nome}</div>
                </div>
              </div>
              <div className="mt-4 text-sm text-zinc-400">
                Nota: Em modo Real, métricas deverão vir de ClickHouse/Superset; esta secção mantém dados de referência locais.
              </div>
            </Card>
          )}
        </div>

        {/* Sidebar */}
        <div className="grid gap-6">
          <Card title="Sessão & Integração">
            <div className="grid gap-2 text-sm">
              <div><span className="text-zinc-400">Utilizador:</span> {logged? session.name : "Anónimo"}</div>
              <div className="text-xs text-zinc-500">Método: {logged? session.method : "—"}</div>
              <label className="text-xs text-zinc-400 mt-2">Base URL (modo Real)</label>
              <input value={baseURL} onChange={e=>setBaseURL(e.target.value)} className="bg-zinc-900 border border-zinc-700 rounded-xl px-3 py-2 text-sm" placeholder="http://localhost:8080" />
              <div className="text-xs text-zinc-500">Endpoints esperados: /v1/services, /v1/sites, /v1/slots, /v1/appointments</div>
            </div>
          </Card>

          <Card title="Estado do sistema">
            <div className="grid grid-cols-2 gap-2 text-sm">
              <div className="rounded-lg bg-zinc-900 border border-zinc-700 p-2">Keycloak
                <div className={`text-xs ${logged? 'text-emerald-400':'text-amber-300'}`}>{logged? 'sessão (simulada)' : 'não autenticado'}</div>
              </div>
              <div className="rounded-lg bg-zinc-900 border border-zinc-700 p-2">Gateway
                <div className={`text-xs ${mode==='real'?'text-emerald-400':'text-amber-300'}`}>{mode==='real'?'real (configurado)':'demo'}</div>
              </div>
              <div className="rounded-lg bg-zinc-900 border border-zinc-700 p-2">APIs
                <div className="text-amber-300 text-xs">{mode==='real'?'real (com fallback)':'demo'}</div>
              </div>
              <div className="rounded-lg bg-zinc-900 border border-zinc-700 p-2">PostgreSQL
                <div className="text-emerald-400 text-xs">operacional (simulado)</div>
              </div>
            </div>
          </Card>

          <Card title="Acessibilidade (WCAG 2.2 AA)">
            <ul className="list-disc list-inside text-sm text-zinc-300 space-y-1">
              <li>Contraste ≥ 4.5:1.</li>
              <li>Navegação por teclado e focos visíveis.</li>
              <li>ARIA adequada nos controlos e labels.</li>
            </ul>
          </Card>
        </div>
      </main>

      <footer className="max-w-6xl mx-auto px-4 pb-10 pt-4 text-xs text-zinc-500">
        PUE · Demo de Front-End · React + Tailwind (CDN) · © Estado Português
      </footer>

      {/* Modal de Login Simulado */}
      <Modal open={loginOpen} title="Entrar — OIDC (simulado)" onClose={()=>setLoginOpen(false)}>
        <div className="grid gap-3">
          <div className="grid gap-1">
            <label className="text-sm text-zinc-300">Nome</label>
            <input value={loginName} onChange={e=>setLoginName(e.target.value)} className="bg-zinc-900 border border-zinc-700 rounded-xl px-3 py-2" placeholder="Nome completo" />
          </div>
          <div className="grid gap-1">
            <label className="text-sm text-zinc-300">E-mail</label>
            <input value={loginEmail} onChange={e=>setLoginEmail(e.target.value)} className="bg-zinc-900 border border-zinc-700 rounded-xl px-3 py-2" placeholder="email@exemplo.pt" />
          </div>
          <div className="grid gap-1">
            <label className="text-sm text-zinc-300">Método</label>
            <select value={loginMethod} onChange={e=>setLoginMethod(e.target.value)} className="bg-zinc-900 border border-zinc-700 rounded-xl px-3 py-2">
              <option>CC/Chave Móvel (simulado)</option>
              <option>eIDAS (simulado)</option>
            </select>
          </div>
          <div className="flex justify-end gap-2 mt-2">
            <button onClick={()=>setLoginOpen(false)} className="px-3 py-2 rounded-xl bg-zinc-800 text-zinc-200">Cancelar</button>
            <button onClick={doLogin} className="px-3 py-2 rounded-xl bg-[#ffcc66] text-black">Entrar</button>
          </div>
        </div>
      </Modal>
    </div>
  );
}

document.addEventListener("DOMContentLoaded", () => {
  const root = ReactDOM.createRoot(document.getElementById("root"));
  root.render(<App/>);
});
